<?php

namespace App\Http\Controllers;

use App\Models\Entity\Cliente;
use App\Models\Entity\TipoPagamento;
use App\Models\Facade\ClienteDB;
use App\Models\Facade\FinanceiroDB;
use App\Models\Regras\FinanceiroRegras;
use DateInterval;
use DateTime;
use Exception;
use Illuminate\Http\Request;

class FinanceiroController extends Controller
{
    public function historico()
    {
        $anoSelec = request('ano');
        $mesSelec = request('mes');
        $cliente   = request('cliente');
        $clientes  = ClienteDB::todos();
        $anoRef    = [2021, 2022];
        $meses     = todosMeses();
        $cobrancas = [];
        $emAtraso  = false;
        $diasEmAtraso = 0;
        $proximaDataAPagar = '';

        if($cliente){
            $cobrancas = FinanceiroDB::cobrancas($cliente, $anoSelec, $mesSelec);

            if($cobrancas->count() > 0) {
                $emAtraso = FinanceiroRegras::verificaAtraso($cliente);
                
                if($emAtraso){
                    $diasEmAtraso = FinanceiroRegras::getDiasEmAtraso();
                    $proximaDataAPagar = FinanceiroRegras::getProximaDataAPagarFormat();
                }
            }
        }

        return view('financeiro.historico', compact('clientes', 'anoSelec', 'mesSelec', 'cliente', 'cobrancas', 'anoRef', 'meses', 'emAtraso', 'diasEmAtraso', 'proximaDataAPagar'));
    }

    public function create($id_cliente)
    {
        $formasPagamento = TipoPagamento::whereNotIn('id', [4])->get();
        $cliente         = ClienteDB::find($id_cliente);
        $anoRef          = [2021, 2022];
        $meses           = todosMeses();

        $cliente->valorDependentes = FinanceiroRegras::getValorDependentes($cliente->dependentes);
        $cliente->valorCliente     = FinanceiroRegras::getValorMensalidadeTitular($cliente->dependentes);
        $cliente->valorCobranca    = FinanceiroRegras::getValorTotalCobranca($cliente->valorCliente, $cliente->valorDependentes);

        return view('financeiro.create', compact('cliente', 'anoRef', 'meses', 'formasPagamento'));
    }

    public function edit($id)
    {
        $cobranca        = FinanceiroDB::cobranca($id);
        $formasPagamento = TipoPagamento::whereNotIn('id', [4])->get();
        $cliente         = ClienteDB::find($cobranca->fk_cliente);
        $anoRef          = [2021, 2022];
        $meses           = todosMeses();

        return view('financeiro.edit', compact('cobranca', 'cliente', 'anoRef', 'meses', 'formasPagamento'));
    }

    public function store(Request $request)
    {
        //Validação de Pagamento
        $ultimaCobranca = FinanceiroDB::getUltimaCobrancaPaga($request->id_cliente);

        if($ultimaCobranca) {

            $proximoMesAPagar = ($ultimaCobranca->mes_ref != 12) ? ($ultimaCobranca->mes_ref + 1) : '01'; //janeiro

            $proximoAnoAPagar = ($proximoMesAPagar != '01' ? $ultimaCobranca->ano_ref : ($ultimaCobranca->ano_ref+1));

            if(FinanceiroDB::isCobrancaPaga($request->id_cliente, $request->ano, $request->mes)) {
                return response()->json(['error' => "<b>Atenção</b>, já existe um registro de pagamento para o mês informado.<br />O próximo mês a ser pago é <b>".mesPorExtenso($proximoMesAPagar)."/".$proximoAnoAPagar."</b>. Altere o mês de referência."], 500);
            }

            if($proximoMesAPagar != $request->mes) {
                return response()->json(['error' => "O próximo mês a ser pago é <b>".mesPorExtenso($proximoMesAPagar)."/".$proximoAnoAPagar."</b>. Altere o mês de referência."], 500);
            }
        }
        //end validação de Pagamento

        try {
            FinanceiroRegras::salvar($request);
            $request->session()->flash('sucesso', 'Dados Financeiros salvos com sucesso.');
            return response()->json();
        }
        catch (Exception $ex){
            return response()->json(['error' => "Algo ocorreu ao registrar os dados de pagamento.<br />".$ex->getMessage()], 500);
        }
    }

    public function remove(Request $request)
    {
        try {
            FinanceiroRegras::removePagamento($request->id);
            $request->session()->flash('sucesso', 'O registro de pagamento foi removido com sucesso.');
            return response()->json();
        }
        catch (Exception $ex){
            return response()->json(['error' => "Erro ao remover o pagamento.<br />".$ex->getMessage()], 500);
        }
    }

    public function relatorio()
    {

    }

    public function relatorioPagamentosEmAtraso()
    {
        $clientes = ClienteDB::todos();

        if($clientes->count()){
            foreach($clientes as $i => $c) {
                $ultimaCobranca = FinanceiroDB::getUltimaCobrancaPaga($c->id);

                //calcular o valor a pagar
                if($clientes[$i]->dependentes > config('parque.limite_dependentes_sem_custo')){
                    $dependentesAdicionais = $clientes[$i]->dependentes - config('parque.limite_dependentes_sem_custo');
                    $valorDependenteAdicional = config('parque.valor_dependente_adicional') * $dependentesAdicionais;
                    $clientes[$i]->valor = config('parque.valor_mensalidade_com_dependente') + $valorDependenteAdicional;
                }else {
                    $clientes[$i]->valor = config('parque.valor_mensalidade');
                }

                if(!$ultimaCobranca) {
                    $clientes[$i]->atraso = true;
                    $clientes[$i]->dtUltimoVencimento = null;
                    continue;
                }


                $dtCorrente = new DateTime(date('Y-m-d'));
                $dtUltimoVencimento = new DateTime($ultimaCobranca->dt_vencimento);
                $dtProximoVencimento =  $dtUltimoVencimento->add(new DateInterval('P30D'));

                if($dtCorrente > $dtProximoVencimento) {
                    $clientes[$i]->atraso = true;
                    $clientes[$i]->dtUltimoVencimento = $dtProximoVencimento->format('d/m/Y');
                }else {
                    unset($clientes[$i]);
                }
            }
        }

        return view('financeiro.relatorio-pagamento-atraso', compact('clientes'));
    }

    public function relatorioPagamentos()
    {
        $mes = (request('mes') ?? date('m'));
        $ano = (request('ano') ?? date('Y'));

        $pagamentos = FinanceiroDB::cobrancas(null, $ano, $mes);

        

        return view('financeiro.relatorio-pagamento', compact('pagamentos', 'mes', 'ano'));
    }
}
